/*
   (c) Copyright 2002-2003  Denis Oliver Kropp <dok@directfb.org>
   All rights reserved.

   XDirectFB is mainly based on XDarwin and
   also contains some KDrive, XFree and XWin code.
*/
/**************************************************************
 *
 * DirectFB-specific support for the DirectFB X Server
 *
 * By Gregory Robert Parker
 *
 **************************************************************/
/* $XFree86: xc/programs/Xserver/hw/directfb/bundle/directfb.c,v 1.25 2002/02/17 03:15:19 torrey Exp $ */

#include "directfbScreen.h"
#include "directfbX.h"
#include "directfbClipboard.h"
#include "directfbCursor.h"
#include "directfbDGA.h"
#include "rootlessDirectFB.h"

/* X headers */
#include "X.h"
#include "Xproto.h"
#include "os.h"
#include "servermd.h"
#include "inputstr.h"
#include "scrnintstr.h"
#include "mibstore.h"
#include "mipointer.h"
#include "micmap.h"
#include "fb.h"
#include "site.h"
#include "globals.h"
#include "dix.h"

/* System headers */
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Shared global variables for DirectFB modes */
int                     directfbScreenIndex = 0;
IDirectFB              *dfb                 = NULL;
IDirectFBEventBuffer   *dfbEvents           = NULL;
IDirectFBInputDevice   *dfbKeyboard         = NULL;


static struct {
     DFBSurfacePixelFormat format;
     unsigned int          depth;
     unsigned int          bitsPerRGB;
     unsigned int          redMask;
     unsigned int          greenMask;
     unsigned int          blueMask;
} directfbFormats[] = {
     {
          DSPF_RGB332,
          8,
          3,
          0x000000E0,
          0x0000001C,
          0x00000003
     },
     {
          DSPF_ARGB1555,
          15,
          5,
          0x00007C00,
          0x000003E0,
          0x0000001F
     },
     {
          DSPF_RGB16,
          16,
          6,
          0x0000F800,
          0x000007E0,
          0x0000001F
     },
     {
          DSPF_RGB24,
          24,
          8,
          0x00FF0000,
          0x0000FF00,
          0x000000FF
     },
     {
          DSPF_RGB32,
          24,
          8,
          0x00FF0000,
          0x0000FF00,
          0x000000FF
     },
     {
          DSPF_ARGB,
          24,
          8,
          0x00FF0000,
          0x0000FF00,
          0x000000FF
     }
};

static const int NUMDFBFORMATS = sizeof(directfbFormats)/sizeof(directfbFormats[0]);


/* Thread that fetches events from DirectFB and writes them into the pipe */
static pthread_t eventsThread = -1;

static void *
XDirectFBEventsThread (void *arg)
{
     DFBEvent event;

     while (TRUE) {
          dfbEvents->WaitForEvent (dfbEvents);

          while (dfbEvents->GetEvent (dfbEvents, &event) == DFB_OK) {
               if (event.clazz == DFEC_USER)
                    return NULL;

               write (directfbEventsPipe[1], &event, sizeof(DFBEvent));
          }
     }

     return NULL;
}

/*
===========================================================================

 Screen functions

===========================================================================
*/

/*
 * XDirectFBSaveScreen
 *  X screensaver support. Not implemented.
 */
static Bool
XDirectFBSaveScreen(ScreenPtr pScreen, int on)
{
     /* FIXME */
     if (on == SCREEN_SAVER_FORCER) {
     }
     else if (on == SCREEN_SAVER_ON) {
     }
     else {
     }

     return TRUE;
}

static void
XDirectFBBlockHandler (int     nScreen,
                       pointer pBlockData,
                       pointer pTimeout,
                       pointer pReadMask)
{
     DirectFBScreenPtr priv = DIRECTFB_PRIV(screenInfo.screens[nScreen]);

     /* Signal threaded modules to begin */
     if (priv != NULL && !priv->serverRunning) {
          int iReturn;

          /* Flag that modules are to be started */
          priv->serverRunning = TRUE;

          /* Unlock the mutex for threaded modules */
          iReturn = pthread_mutex_unlock (&priv->serverStarted);
          if (iReturn != 0) {
               ErrorF ("XDirectFBBlockHandler - "
                       "pthread_mutex_unlock () failed: %d\n", iReturn);
          }
     }
}

static Bool
XDirectFBCloseScreen (int index, ScreenPtr pScreen)
{
     Bool                   ret;
     DirectFBScreenPtr      priv  = DIRECTFB_PRIV(pScreen);
     IDirectFBDisplayLayer *layer = priv->layer;

     /* Unwrap CloseScreen and call the original function. */
     pScreen->CloseScreen = priv->CloseScreen;
     ret = (*pScreen->CloseScreen) (index, pScreen);

     /* Release our interface to the display layer of this screen. */
     layer->Release (layer);

     pthread_mutex_destroy( &priv->serverStarted );

     /* Free space of private per screen DirectFB specific storage. */
     xfree ((pointer) priv);

     return ret;
}

/*
 * XDirectFBAddScreen
 *  This is a callback from dix during AddScreen() from InitOutput().
 *  Initialize the screen and communicate information about it back to dix.
 */
Bool
XDirectFBAddScreen (int         index,
                    ScreenPtr   pScreen,
                    int         argc,
                    char      **argv)
{
     DFBResult             ret;
     int                   i, dpi;
     VisualPtr             pvisual;
     DirectFBScreenPtr     priv;

     /* Allocate space for private per screen DirectFB specific storage */
     priv = xcalloc(sizeof(DirectFBScreenRec), 1);
     DIRECTFB_PRIV(pScreen) = priv;

     /* Get an interface to the layer for this screen */
     ret = dfb->GetDisplayLayer (dfb, directfbPrefs.layerID, &priv->layer);
     if (ret) {
          DirectFBError ("XDirectFBAddScreen: GetDisplayLayer", ret);
          return FALSE;
     }

     /* Enter administrative cooperative level */
     ret = priv->layer->SetCooperativeLevel (priv->layer,
                                             DLSCL_ADMINISTRATIVE);
     if (ret) {
          ErrorF ("IDirectFBDisplayLayer::SetCooperativeLevel() failed (%s)\n",
                  DirectFBErrorString (ret));
          return FALSE;
     }

     /* Get layer configuration (size, format, etc.) */
     priv->layer->GetConfiguration (priv->layer, &priv->layer_config);

     /* Find primary format */
     for (i=0; i<NUMDFBFORMATS; i++)
          if (directfbFormats[i].format == priv->layer_config.pixelformat)
               break;

     if (i >= NUMDFBFORMATS) {
          ErrorF ("XDirectFBAddScreen: Unknown DirectFB Pixelformat 0x%08x, "
                  "trying RGB16... ", priv->layer_config.pixelformat);

          priv->layer_config.pixelformat = DSPF_RGB16;

          /* Set modified layer configuration */
          ret = priv->layer->SetConfiguration (priv->layer,
                                               &priv->layer_config);
          if (ret) {
               ErrorF ("FAILED (%s)\n", DirectFBErrorString (ret));
               return FALSE;
          }

          ErrorF ("OK\n");

          /* Find new primary format */
          for (i=0; i<NUMDFBFORMATS; i++)
               if (directfbFormats[i].format == priv->layer_config.pixelformat)
                    break;
     }


     /* Reset the visual list. */
     miClearVisualTypes();

     /* Set primary format */
     if (!miSetVisualTypes (directfbFormats[i].depth, TrueColorMask,
                            directfbFormats[i].bitsPerRGB, TrueColor)) {
          ErrorF("XDirectFBAddScreen: miSetVisualTypes failed!\n");

          return FALSE;
     }

     miSetPixmapDepths();

     if (monitorResolution)
          dpi = monitorResolution;
     else
          dpi = 75;

     /* Initialize fb */
     if (! fbScreenInit (pScreen,
                         NULL,                      /* pointer to screen bitmap */
                         priv->layer_config.width,
                         priv->layer_config.height, /* screen size in pixels */
                         dpi, dpi,                  /* dots per inch */
                         priv->layer_config.width,  /* pixel width of framebuffer */
                         DFB_BYTES_PER_PIXEL(priv->layer_config.pixelformat) * 8))
     /* bits per pixel for screen */
     {
          ErrorF("XDirectFBAddScreen: fbScreenInit failed!\n");

          return FALSE;
     }

     /* Set the RGB order correctly. */
     for (i=0, pvisual = pScreen->visuals; i < pScreen->numVisuals; i++, pvisual++) {
          if (pvisual->class == TrueColor || pvisual->class == DirectColor) {
               switch (pvisual->bitsPerRGBValue) {
                    case 3:
                         pvisual->offsetRed   = 5;
                         pvisual->offsetGreen = 2;
                         pvisual->offsetBlue  = 0;
                         pvisual->redMask     = 0xe0;
                         pvisual->greenMask   = 0x1c;
                         pvisual->blueMask    = 0x03;
                         break;
                    case 5:
                         pvisual->offsetRed   = 10;
                         pvisual->offsetGreen =  5;
                         pvisual->offsetBlue  =  0;
                         pvisual->redMask     = 0x7c00;
                         pvisual->greenMask   = 0x03e0;
                         pvisual->blueMask    = 0x001f;
                         break;
                    case 6:
                         pvisual->offsetRed   = 11;
                         pvisual->offsetGreen =  5;
                         pvisual->offsetBlue  =  0;
                         pvisual->redMask     = 0xf800;
                         pvisual->greenMask   = 0x07e0;
                         pvisual->blueMask    = 0x001f;
                         break;
                    case 8:
                         pvisual->offsetRed   = 16;
                         pvisual->offsetGreen =  8;
                         pvisual->offsetBlue  =  0;
                         pvisual->redMask     = 0xff0000;
                         pvisual->greenMask   = 0x00ff00;
                         pvisual->blueMask    = 0x0000ff;
                         break;
                    default:
                         ErrorF("XDirectFBAddScreen: unknown bitsPerRGBValue (%d)!\n",
                                pvisual->bitsPerRGBValue);
                         break;
               }
          }
     }

#ifdef RENDER
     if (! fbPictureInit(pScreen, 0, 0))
          return FALSE;
#endif

#ifdef MITSHM
     ShmRegisterFbFuncs(pScreen);
#endif

#ifdef XFreeXDGA
     XDirectFBDGAInit(pScreen);
#endif

     /* This must be initialized (why doesnt X have a default?) */
     pScreen->SaveScreen = XDirectFBSaveScreen;

     pScreen->BlockHandler = XDirectFBBlockHandler;

     /* Wrap CloseScreen */
     priv->CloseScreen = pScreen->CloseScreen;
     pScreen->CloseScreen = XDirectFBCloseScreen;

     /* Setup rootless X support */
     if (! XDirectFBRootlessSetupScreen(index, pScreen))
          return FALSE;

     /* Setup cursor support */
     if (! XDirectFBInitCursor(pScreen))
          return FALSE;

     /* Create and install the default colormap */
     if (! fbCreateDefColormap( pScreen ))
          return FALSE;

     dixScreenOrigins[index].x = 0;
     dixScreenOrigins[index].y = 0;

     ErrorF("Screen %d added: %dx%d\n", index,
            priv->layer_config.width, priv->layer_config.height);

     pthread_mutex_init( &priv->serverStarted, NULL );
     pthread_mutex_lock( &priv->serverStarted );

     /* Initialize the clipboard manager */
     if (!XDirectFBClipboardInit (pScreen)) {
          ErrorF ("XDirectFBAddScreen - XDirectFBClipboardInit () failed.\n");
          return FALSE;
     }

     return TRUE;
}

/*
 * XDirectFBInitOutput
 *  DirectFB display initialization.
 */
Bool
XDirectFBInitOutput (int    argc,
                     char **argv)
{
     DFBResult ret;
     static unsigned long generation = 0;

     /* Allocate private storage for each screens DirectFB specific info */
     if (generation != serverGeneration) {
          directfbScreenIndex = AllocateScreenPrivateIndex();
          generation = serverGeneration;
     }

     /* Initialize DirectFB if this is the first time */
     if (!dfb) {
          /* Pass command line for parsing */
          ret = DirectFBInit (&argc, &argv);
          if (ret) {
               DirectFBError ("XDirectFBInitOutput: DirectFBInit", ret);
               return FALSE;
          }

          /* SIGTERM is handled by the generic X server code */
          DirectFBSetOption ("dont-catch", "2, 15");

          /* Create the super interface */
          ret = DirectFBCreate (&dfb);
          if (ret) {
               DirectFBError ("XDirectFBInitOutput: DirectFBCreate", ret);
               return FALSE;
          }

          /* Get the primary keyboard */
          if (dfb->GetInputDevice (dfb, DIDID_KEYBOARD, &dfbKeyboard) == DFB_OK) {
               /* Get the current LED state */
               dfbKeyboard->GetLockState (dfbKeyboard, &directfbLocks);
          }

          /*
           * Create an event buffer that will be attached to each window.
           * Start with input devices connected for global key events.
           */
          dfb->CreateInputEventBuffer (dfb, DICAPS_ALL, DFB_TRUE, &dfbEvents);

          /*
           * Create the event thread that waits idle for events
           * and writes input and window events into the event pipe.
           */
          pthread_create (&eventsThread, NULL, XDirectFBEventsThread, NULL);
     }

     return TRUE;
}

/*
 * XDirectFBCloseOutput
 *  DirectFB display deinitialization.
 */
void
XDirectFBCloseOutput()
{
     DFBEvent ev;

     /* A minimalistic user event */
     ev.clazz = DFEC_USER;

     /* Post the event to the event thread */
     dfbEvents->PostEvent (dfbEvents, &ev);

     /* Wait until the event thread received the event and returned */
     pthread_join (eventsThread, NULL);

     /* Release event buffer */
     dfbEvents->Release (dfbEvents);
     dfbEvents = NULL;

     /* Release the keyboard */
     if (dfbKeyboard) {
          dfbKeyboard->Release (dfbKeyboard);
          dfbKeyboard = NULL;
     }

     /* Make sure all windows are destroyed. */
     XDirectFBDestroyAll();

     /* Release the super interface (will likely do a DirectFB shutdown) */
     dfb->Release (dfb);
     dfb = NULL;
}

