/*
 * Copyright 2050 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.drools.guvnor.client.packages;


import org.drools.guvnor.client.common.FormStyleLayout;
import org.drools.guvnor.client.common.FormStylePopup;
import org.drools.guvnor.client.common.GenericCallback;
import org.drools.guvnor.client.common.InfoPopup;
import org.drools.guvnor.client.messages.Constants;
import org.drools.guvnor.client.resources.Images;
import org.drools.guvnor.client.rpc.PackageConfigData;
import org.drools.guvnor.client.rpc.RepositoryServiceFactory;
import org.drools.guvnor.client.ruleeditor.MultiViewRow;
import org.drools.guvnor.client.ruleeditor.VersionChooser;
import org.drools.guvnor.client.rulelist.OpenItemCommand;

import org.drools.guvnor.client.widgets.tables.DependenciesPagedTable;
import org.drools.guvnor.client.widgets.tables.DependenciesPagedTableReadOnly;

import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Style.FontWeight;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;

/**
 * This is the widget for building dependencies. 
 */
public class DependencyWidget extends Composite {

    private static Images         images    = (Images) GWT.create( Images.class );
    private Constants             constants = ((Constants) GWT.create( Constants.class ));

    private FormStyleLayout        layout;
    private DependenciesPagedTable table;

    private PackageConfigData conf;
    private boolean isHistoricalReadOnly = false;
    
    public DependencyWidget(final PackageConfigData conf, boolean isHistoricalReadOnly) {
        this.conf = conf;
        this.isHistoricalReadOnly = isHistoricalReadOnly;
        layout = new FormStyleLayout();
        
        VerticalPanel header = new VerticalPanel();
        Label caption = new Label( "Dependencies" );
        caption.getElement().getStyle().setFontWeight( FontWeight.BOLD );
        header.add( caption );
        header.add( dependencyTip() );

        layout.addAttribute( "",
                header );

/*        layout.addHeader( images.statusLarge(),
                      header );*/

        VerticalPanel vp = new VerticalPanel();
        vp.setHeight( "100%" );
        vp.setWidth( "100%" );

        //pf.startSection();
		layout.addRow(vp);

		table = new DependenciesPagedTable(conf.uuid, new OpenItemCommand() {
			public void open(String path) {
				showEditor(path);
			}

			public void open(MultiViewRow[] rows) {
				// Do nothing, unsupported
			}
		});

		layout.addRow(table);

		initWidget(layout);
    }

    private Widget dependencyTip() {
        HorizontalPanel hp = new HorizontalPanel();
        hp.add( new HTML( "<small><i>"
                          + "This shows exact versions of assets that this package contains."
                          + "</i></small>" ) );
        InfoPopup pop = new InfoPopup( "Edit Dependency",
                "Edit dependency version to build a package against specific versions of assets" );
        hp.add( pop );
        return hp;
    }
    
    public static String[] decodeDependencyPath(String dependencyPath) {
    	if(dependencyPath.indexOf("?version=") >=0) {
    		return dependencyPath.split("\\?version=");
    	} else {
    		return new String[]{dependencyPath, "LATEST"};
    	}
    }
    
    public static String encodeDependencyPath(String dependencyPath, String dependencyVersion) {
    	return dependencyPath + "?version=" + dependencyVersion;
    }
       
    private void showEditor(final String dependencyPath) {
		final FormStylePopup editor = new FormStylePopup(images.management(), "Edit Dependency");
/*		editor.addRow(new HTML("<i>" + "Choose the version you want to depend on"
				+ "</i>"));
*/
		editor.addAttribute("Dependency Path: ", new Label(decodeDependencyPath(dependencyPath)[0]));
		final VersionChooser versionChoose = new VersionChooser( 
				decodeDependencyPath(dependencyPath)[1],
				conf.uuid,
				decodeDependencyPath(dependencyPath)[0],
                new Command() {
                    public void execute() {
                        table.refresh();                        
                    }				    
				});
		editor.addAttribute("Dependency Version: ",  versionChoose);

		HorizontalPanel hp = new HorizontalPanel();
		Button useSelectedVersionButton = new Button("Use selected version"); 
		hp.add(useSelectedVersionButton);
        useSelectedVersionButton.addClickHandler(new ClickHandler() {
            public void onClick(ClickEvent w) {
                String selectedVersion = versionChoose.getSelectedVersionName();
                if(selectedVersion == null) {
                	return;
                }
                if (Window.confirm("Are you sure you want to use version: " + selectedVersion  + " as dependency?")) {
                    RepositoryServiceFactory.getPackageService().updateDependency(
                            conf.uuid,
                            encodeDependencyPath(DependencyWidget
                                    .decodeDependencyPath(dependencyPath)[0],
                                    selectedVersion),
                            new GenericCallback<Void>() {
                                public void onSuccess(Void v) {
                                    editor.hide();
                                    table.refresh();
                                }
                            });
                }
            }
        });
        useSelectedVersionButton.setEnabled(!isHistoricalReadOnly);
		
		Button cancel = new Button(constants.Cancel());
		hp.add(cancel);
		cancel.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent w) {
				editor.hide();
			}
		});
		
		editor.addAttribute("", hp);
		editor.show();
	}
}
