/*
 * DB: Database Selection Form
 * Copyright (C) 1998  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "callback.h"
#include "db.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

typedef struct
{
    Char name[dmDBNameLength+1];
    UInt cardNo;
    LocalID dbID;
} ChooserItemType;

typedef ChooserItemType * ChooserItemPtr;

static ChooserItemPtr items = 0;
static Word NumDatabases, TopVisibleDatabase;

static void
EnumerateDatabases(void)
{
    DmSearchStateType state;
    UInt cardNo, i;
    LocalID dbID;
    Boolean flag;
    
    /* Count the number of databases present. */
    NumDatabases = 0;
    flag = true;
    while (DmGetNextDatabaseByTypeCreator(flag, &state, DBTypeID, 0,
					  false, &cardNo, &dbID) == 0) {
	flag = false;
	NumDatabases++;
    }

    /* Allocate space for the database info structure. */
    if (items)
	MemPtrFree(items);
    items = MemPtrNew(NumDatabases * sizeof(ChooserItemType));

    /* Copy the information into place. */
    i = 0;
    flag = true;
    while (i < NumDatabases &&
	   DmGetNextDatabaseByTypeCreator(flag, &state, DBTypeID, 0,
					  false, &cardNo, &dbID) == 0) {
	DmDatabaseInfo(cardNo, dbID, items[i].name, 0, 0, 0, 0, 0,
		       0, 0, 0, 0, 0);
	items[i].cardNo = cardNo;
	items[i].dbID = dbID;

	flag = false;
	i++;
    }
}

static void
UpdateScrollers(void)
{
    Word upIndex, downIndex, dbIndex;
    Boolean scrollUp, scrollDown;
    SWord row;
    FormPtr form;
    TablePtr table;

    scrollUp = TopVisibleDatabase != 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_Chooser_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1) {
	dbIndex = TblGetRowID(table, row);
	scrollDown = dbIndex < NumDatabases - 1;
    } else {
	scrollDown = false;
    }

    upIndex = FrmGetObjectIndex(form, ctlID_Chooser_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_Chooser_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void
DrawChooserItem(VoidPtr table, Word row, Word col, RectanglePtr bounds)
{
    Word index;

    CALLBACK_PROLOGUE;
    index = TblGetRowID(table, row);
    WinDrawChars(items[index].name, StrLen(items[index].name),
		 bounds->topLeft.x, bounds->topLeft.y);
    CALLBACK_EPILOGUE;
}

static void
LoadTable(TablePtr table)
{
    Word numRows, row;
    DmSearchStateType state;
    UInt cardNo;
    LocalID dbID;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++) {
	if (TopVisibleDatabase + row < NumDatabases) {
	    TblSetRowID(table, row, TopVisibleDatabase + row);
	    TblSetItemStyle(table, row, 0, customTableItem);
	    TblSetRowUsable(table, row, true);
	    TblMarkRowInvalid(table, row);
	} else {
	    TblSetRowUsable(table, row, false);
	}
    }

    UpdateScrollers();
}

static void
InitTable(TablePtr table)
{
    TblSetCustomDrawProcedure(table, 0, DrawChooserItem);
    TblSetColumnUsable(table, 0, true);
    LoadTable(table);
}

static void
Scroll(DirectionType direction)
{
    UInt newTopVisibleDatabase;

    newTopVisibleDatabase = TopVisibleDatabase;

    if (direction == down) {
	newTopVisibleDatabase++;
    } else {
	newTopVisibleDatabase--;
    }

    if (TopVisibleDatabase != newTopVisibleDatabase) {
	FormPtr form;
	TablePtr table;

	TopVisibleDatabase = newTopVisibleDatabase;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_Chooser_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

Boolean
ChooserHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    Word index;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_Chooser_Table);
	TopVisibleDatabase = 0;
	EnumerateDatabases();
	InitTable(table);
	CtlSetValue(GetObjectPtr(form, ctlID_Chooser_DelButton), 0);
	FrmDrawForm(form);
	return true;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_Chooser_Table);
	EnumerateDatabases();
	InitTable(table);
	CtlSetValue(GetObjectPtr(form, ctlID_Chooser_DelButton), 0);
	FrmEraseForm(form);
	FrmDrawForm(form);
	return true;

    case frmCloseEvent: {
	CALLBACK_PROLOGUE; /* XXX Work around the gcc %a4 bug */
	if (items) {
	    MemPtrFree(items);
	    items = 0;
	}
	CALLBACK_EPILOGUE;
	break;
    }

    case tblSelectEvent:
	index = TblGetRowID(event->data.tblSelect.pTable,
			    event->data.tblSelect.row);
	form = FrmGetActiveForm();
	if (CtlGetValue(GetObjectPtr(form, ctlID_Chooser_DelButton))) {
	    TblUnhighlightSelection(GetObjectPtr(form, ctlID_Chooser_Table));
	    if (FrmCustomAlert(alertID_ConfirmDatabaseDelete,
			       items[index].name, " ", " ") == 0) {
		DmDeleteDatabase(items[index].cardNo, items[index].dbID);
		FrmUpdateForm(formID_Chooser, 0);
	    }
	    CtlSetValue(GetObjectPtr(form, ctlID_Chooser_DelButton), 0);
	} else {
	    OpenDatabase(items[index].cardNo, items[index].dbID);
	    FrmGotoForm(formID_ListView);
	}
	return true;
					      
    case ctlSelectEvent:
	if (event->data.ctlSelect.controlID == ctlID_Chooser_NewButton) {
	    DesignNewDB = true;
	    FrmGotoForm(formID_DesignView);
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_Chooser_UpButton:
	    Scroll(up);
	    break;
	case ctlID_Chooser_DownButton:
	    Scroll(down);
	    break;
	}
	break;
    }

    return false;
}
