
/*
 * GNOME Speech - Speech services for the GNOME desktop
 *
 * Copyright 2003 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * dectalkspeaker.c: Implements the DectalkSpeaker object--
 *                 a GNOME Speech driver for Fonix's DECtalk TTS SDK
 *
 */

#include <string.h>
#include <unistd.h>
#include <libbonobo.h>
#include <glib/gmain.h>
#include <glib/gthread.h>
#include <gnome-speech/gnome-speech.h>
#include "dectalkspeaker.h"
 

typedef struct {
    GNOME_Speech_speech_callback_type type;
    gint text_id;
    gint offset;
} index_queue_entry;


static GObjectClass *parent_class;

extern struct voiceinfo voices[MAXVOICES];

/* Unique text id for each utterance spoken by this speaker */
static gint text_id = 0;


static void
dectalk_speaker_add_index(DectalkSpeaker *s,
                          GNOME_Speech_speech_callback_type type,
                          gint text_id,
                          gint offset)
{
    index_queue_entry *e;

    if (s->callback == CORBA_OBJECT_NIL) {
        return;
    }
    
    e = g_new(index_queue_entry, 1);
    e->type = type;
    e->text_id = text_id;
    e->offset = offset;
    s->index_queue = g_slist_append(s->index_queue, e);

    TextToSpeechSpeak (s->handle,
		       " [:index mark 00]", TTS_FORCE);
}


static gboolean
dectalk_timeout (void *data)
{
	DectalkSpeaker *s = DECTALK_SPEAKER 
(data);
	CORBA_Environment ev;
	
	if (s->callback != CORBA_OBJECT_NIL && s->post_queue) {
		index_queue_entry *e = (index_queue_entry *) s->post_queue->data;
		s->post_queue = g_slist_remove_link (s->post_queue, s->post_queue);
		CORBA_exception_init (&ev);
		GNOME_Speech_SpeechCallback_notify (s->callback,
						    e->type,
						    e->text_id,
						    e->offset,
						    &ev);
		CORBA_exception_free (&ev);
		g_free (e);
	}
	return TRUE;
}


static void
dectalk_callback (LONG lParam1,
		       LONG lParam2, 
		       DWORD dwCallbackParameter, 
		       UINT uiMsg) 
{
	DectalkSpeaker *s = DECTALK_SPEAKER(dwCallbackParameter);
	index_queue_entry *e = NULL;
	
	switch (uiMsg) {
	case TTS_MSG_INDEX_MARK:
		if (s && s->index_queue) {
			e = (index_queue_entry *) s->index_queue->data;
			s->index_queue = g_slist_remove_link (s->index_queue, s->index_queue);
			s->post_queue = g_slist_append (s->post_queue, e);
		}
	}
	return;
}


static DectalkSpeaker *
dectalk_speaker_from_servant(PortableServer_Servant *servant)
{
    return(DECTALK_SPEAKER(bonobo_object_from_servant(servant)));
}


static void
dectalk_add_parameter(DectalkSpeaker *dectalk_speaker,
                      const gchar *parameter_name,
                      gdouble min,
                      gdouble current,
		      gdouble max,
                      parameter_set_func func)
{
    Speaker *speaker = SPEAKER(dectalk_speaker);

    func (speaker, current);
    speaker_add_parameter(speaker, parameter_name, 
                          min, current, max, func);
}


static CORBA_boolean
dectalk_registerSpeechCallback(PortableServer_Servant servant,
                               const GNOME_Speech_SpeechCallback callback,
                               CORBA_Environment *ev)
{
    DectalkSpeaker *s = dectalk_speaker_from_servant(servant);

    /* Store reference to callback */

    s->callback = CORBA_Object_duplicate(callback, ev);

    /* Start timeout */

    s->timeout_id = g_timeout_add_full (G_PRIORITY_HIGH_IDLE, 100,
					dectalk_timeout, s, NULL);
    return TRUE;
}


static CORBA_long
dectalk_say(PortableServer_Servant servant,
            const CORBA_char *text,
            CORBA_Environment *ev)
{
    MMRESULT result;
    DectalkSpeaker *speaker = dectalk_speaker_from_servant(servant);
 
    g_return_val_if_fail(speaker->handle != NULL, FALSE);

    text_id++;
    dectalk_speaker_add_index(speaker,
                              GNOME_Speech_speech_callback_speech_started,
                              text_id, 0);
    TextToSpeechSpeak(speaker->handle, (char *) text, TTS_FORCE);
    dectalk_speaker_add_index(speaker,
                              GNOME_Speech_speech_callback_speech_ended,
                              text_id, 0);

    return(text_id);
}


static CORBA_boolean
dectalk_stop(PortableServer_Servant servant, CORBA_Environment *ev)
{
    MMRESULT result;
    DectalkSpeaker *speaker = dectalk_speaker_from_servant(servant);

    g_return_val_if_fail(speaker->handle != NULL, FALSE);

    result = TextToSpeechReset(speaker->handle, FALSE);

    return((result == MMSYSERR_NOERROR) ? TRUE : FALSE);
}


static void
dectalk_speaker_init(DectalkSpeaker *speaker)
{
    speaker->handle = NULL;
}


static void
dectalk_speaker_finalize(GObject *obj)
{
    DectalkSpeaker *speaker = DECTALK_SPEAKER(obj);

    if (speaker->handle != NULL) {
        TextToSpeechShutdown(speaker->handle);
    }

    if (speaker->timeout_id > 0)
	    g_source_remove (speaker->timeout_id);
    if (parent_class->finalize) {
        parent_class->finalize(obj);
    }
}


static void
dectalk_speaker_class_init(DectalkSpeaker *klass)
{
    SpeakerClass *class = SPEAKER_CLASS(klass);
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
  
    parent_class = g_type_class_peek_parent(klass);
    object_class->finalize = dectalk_speaker_finalize;

    /* Initialize parent class epv table */

    class->epv.say = dectalk_say;
    class->epv.stop = dectalk_stop;
    class->epv.registerSpeechCallback = dectalk_registerSpeechCallback;
}


BONOBO_TYPE_FUNC(DectalkSpeaker, speaker_get_type(), dectalk_speaker);


static gboolean
dectalk_set_rate(Speaker *speaker, gdouble new_rate)
{
    MMRESULT result;
    DectalkSpeaker *s = DECTALK_SPEAKER(speaker);
    gchar * command_string;
    
    command_string = g_strdup_printf ("[:rate %ld]", (long)new_rate);
    TextToSpeechSpeak (s->handle, command_string, TTS_FORCE);
    g_free (command_string);
    return TRUE;
}



static gboolean
dectalk_set_pitch(Speaker *speaker, gdouble new_pitch)
{
    MMRESULT result;
    DectalkSpeaker *s = DECTALK_SPEAKER(speaker);
    gchar * command_string;
    
    command_string = g_strdup_printf ("[:dv ap %ld]", (long)new_pitch);
    TextToSpeechSpeak (s->handle, command_string, TTS_FORCE);
    g_free (command_string);
    return TRUE;
}



static gboolean
dectalk_set_voice(Speaker *speaker, gdouble new_voice)
{
    MMRESULT result;
    DectalkSpeaker *s = DECTALK_SPEAKER(speaker);
    gchar *command_string;
    char *voice_names[] = {
	    "Paul",
	    "Betty",
	    "Harry",
	    "Frank",
	    "Dennis",
	    "Kit",
	    "Ursula",
	    "Rita",
	    "Wendy"};    
    int i = (int) new_voice-1;

    command_string = g_strdup_printf ("[:name %s]", voice_names[i]);
    TextToSpeechSpeak (s->handle, command_string, TTS_FORCE);
    g_free (command_string);
    return TRUE;
}



DectalkSpeaker *
dectalk_speaker_new(const GNOME_Speech_VoiceInfo *voice_spec)
{
    DectalkSpeaker *speaker;
    UINT devOptions = 0;
    int devNo = (int) WAVE_MAPPER;
    gint i;
    int voice_count = MAXVOICES;

    speaker = g_object_new(DECTALK_SPEAKER_TYPE, NULL);
  
    if (TextToSpeechStartup(&speaker->handle, devNo, devOptions,
                            dectalk_callback, speaker) != MMSYSERR_NOERROR) {
	speaker->handle = NULL;
	return(speaker);
    }

    /* Initialize the index queue */

    speaker->index_queue = speaker->post_queue = NULL;

    /* Add parameters */

    dectalk_add_parameter(speaker, "rate", 75, 200, 600,
                          dectalk_set_rate);  


    dectalk_add_parameter(speaker, "pitch", 50, 100, 400,
                          dectalk_set_pitch);  


    /* If call specified name, set the voice to that name */

    if (voice_spec->name) {
	    char *command_string = g_strdup_printf ("[:name %s]", voice_spec->name);
	    TextToSpeechSpeak (speaker->handle, command_string, TTS_FORCE);
	    g_free (command_string);
    }

    /* Add the voice parameter */

    speaker_add_parameter(SPEAKER(speaker), "voice", 0, 0,
                          voice_count-1, dectalk_set_voice);

    /* Add the voice names to the parameter */

    for (i = 0; i < MAXVOICES; i++) {
        speaker_add_parameter_value_description(SPEAKER(speaker), "voice",
                                       (gdouble) i, voices[i].description);
    }
 
    return(speaker);
}
